#Facharbeit
#Created by:
# ____   ___ ___ ____     ________   ___
#/    \ /   \| | \| |     / /__|  \  | |
#| [] |/  _  | |  | |    / /|_ | | \ | |
#|   < | |_| | |  | |\  / / __|| |\ \| |
#| |\ \\     | \__/ | \/ /| |__| | \ | |
#|_| \_\\___/ \___ / \__/ |_____]|  \__| Bauer
#>>>>><<<<<>>>>><<<<<>>>>><<<<<>>>>><<<<
#+++++++++++++++++++++++++++++++++++++++      by d'Guschdl-arts
#Copyright 2010 Rouven Bauer
#    This file is part of "BonSim - The Bonsai Simulation".
#
#    "BonSim - The Bonsai Simulation" is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    "BonSim - The Bonsai Simulation" is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with "BonSim - the Bonsai-Simulation".  If not, see <http://www.gnu.org/licenses/>.

from tkinter import TclError
from detailview import DetailView
from overview import OverView
from speicher_dialog import MemoryDialog as MD

class View:
    def __init__(self,cb,cb_exit=None,cb_speicher=None,get_speicher=None):
        """Creates a view that is usefull to simulate a Bosai-PC
Input:
cb: func - The function that shall be called if the user does an input.
cb will give back the input in the following way:
cb(board,button)
both are int. Board is between 1 and 6. Its the numer of the board on which the Button was pressed. Count from left-top to bottom-right (like the way you read).
button is the number of the button that is pressed. Count from left-top to bottom-right (like the way you read)."""
        self.prevmpr = None
        self.cb = cb
        self.cbS = cb_speicher
        self.getS = get_speicher
        self.ov = OverView(self._cb_ov,self._cb_speicher)
        self.cbE = cb_exit
        self.ov.protocol('WM_DELETE_WINDOW',self._exit)
        self.dv = DetailView(self._cb_dv,master=self.ov)
        self.ov.update_idletasks()
        self.dv.update_idletasks()
        dvg = self.dv.geometry().split('+')[0].split('x')+self.dv.geometry().split('+')[1:]
        ovg = self.ov.geometry().split('+')[0].split('x')+self.ov.geometry().split('+')[1:]
        self.ov.geometry(ovg[0]+'x'+ovg[1]+'+'+str(int(dvg[0])+int(dvg[2])+6)+'+'+dvg[3])

    def _cb_speicher(self):
        """Intern use only.
Passes the unserinput for the memory via the MemoryDialog to cb_speicher."""
        inhalt = MD.ask(self.ov,self.getS() if self.getS != None else [])
        if self.cbS != None and inhalt != None:
            self.cbS(inhalt)

    def speicherstate(self,level):
        """If level is False the MenuEntry "Speicher" will be disabled.
If True the Entry will be enabled."""
        self.ov.menu.entryconfig(0,state='normal' if level else 'disable')
        
    def _cb_ov(self,board):
        "Intern function.\n\
It handels the callback from OverView"
        try: self.dv.show(board)
        except TclError:
            self.dv = DetailView(self._cb_dv,master=self.ov)
            self.dv.show(board)

    def _cb_dv(self,board,part,level):
        "Intern function.\n\
It handels the callback from DetailView and converts the userinput to send it in a callback."
        self.ov.set(board,part,level)
        bpl = (board,part,level)
        d = {(0, 8,1):(0,0),
             (0, 9,1):(0,1),
             (0,10,1):(0,2),
             (0,13,1):(0,3),
             (0,13,0):(0,4),
             (0,14,1):(0,5),
             (0,14,0):(0,6),
             (1,12,1):(1,0),
             (1,12,0):(1,1),
             (1,13,1):(1,2),
             (1,13,0):(1,3),
             (1,14,1):(1,4),
             (1,14,0):(1,5),
             (3,12,1):(3,0),
             (3,12,0):(3,1),
             (3,13,1):(3,2),
             (3,13,0):(3,3),
             (3,14,1):(3,4),
             (3,14,0):(3,5)}
        #HS:
        for i in range(8):
            d[(4,i+1,0)] = (4,i*2)
            d[(4,i+1,1)] = (4,i*2+1)
        d[(4,10,1)] = (4,16)
        d[(4,10,0)] = (4,17)
        d[(4,11,1)] = (4,18)
        d[(4,11,0)] = (4,19)
        d[(4,12,1)] = (4,20)
        d[(4,12,0)] = (4,21)
        d[(4,13,1)] = (4,22)
        d[(4,13,0)] = (4,23)
        d[(4,14,1)] = (4,24)
        d[(4,14,0)] = (4,25)
        if bpl in d.keys():
            if self.cb != None: self.cb(*d[bpl])
    
    def setUp(self,board,level):
        """Set the UP-LED
Input:
board: int - 0 for 'Speicher', 1 for 'PC', 2 for 'BR', 3 for 'Akku' and so on.
level: bool - The level of the Up-LED.

Only exists for 'Akku' and 'PC'."""
        assert board in (1,3)
        self._set(board,10,level)
    def setDown(self,board,level):
        """Set the DOWM-LED
Input:
board: int - 0 for 'Speicher', 1 for 'PC', 2 for 'BR', 3 for 'Akku' and so on.
level: bool - The level of the Down-LED.

Only exists for 'Akku' and 'PC'."""
        assert board in (1,3)
        self._set(board,11,level)
    def setByte1(self,board,value):
        """Set a row of 8 LEDs
Input:
board: int - 0 for 'Speicher', 1 for 'PC', 2 for 'BR', 3 for 'Akku' and so on.
value: int - Between 0 and 255. The Value the Byte of LEDs is set to.

For 'Speicher' Byte1 means the LEDs of the Adressbus
For 'PC' Byte1 means the LEDs of PC's value
For 'BR' Byte1 means the LEDs of BR's value
For 'Akku' Byte1 means the LEDs of BR's value
For 'HS' Byte1 doesn't exist
For 'SW' Byte1 means the first half of the ControllSignal-LEDs"""
        assert board in (0,1,2,3,5)
        Bit0 = {0:7,1:9,2:7,3:9,5:53}[board]
        self._setByte(board,Bit0,value)
        #self.ov.update()
    def setByte2(self,board,value):
        """Set a row of 8 LEDs
Input:
board: int - 0 for 'Speicher', 1 for 'PC', 2 for 'BR', 3 for 'Akku' and so on.
value: int - Between 0 and 255. The Value the Byte of LEDs is set to.

For 'Speicher' Byte1 means the LEDs of the Databus
For 'PC', 'BR', 'Akku' and 'HS' Byte1 doesn't exist
For 'SW' Byte1 means the second half of the ControllSignal-LEDs"""
        assert board in (0,5)
        Bit0 = {0:22,5:61}[board]
        self._setByte(board,Bit0,value)
        #self.ov.update()
    def setHalfByte(self,board,value):
        """Set a row of 5 LEDs
board: int - 0 for 'Speicher', 1 for 'PC', 2 for 'BR', 3 for 'Akku' and so on.
value: int - Between 0 and 15. The Value the Byte of LEDs is set to.

For 'Speicher' and 'PC' HalfByte doesn't exist.
For 'BR' HalfByte means the LEDs of the decoder
For 'Akku' and 'HS' HalfByte doesn't exist
For 'SW' HalfByte means value of the MPC"""
        assert board in (2,5)
        Bit0 = {2:14,5:44}[board]
        self._setByte(board,Bit0,value,half=True)
        #self.ov.update()

    def setMProw(self,row):
        """Set the rows of the MiniProgramm-LEDs.
All other Rows will be turned off automtically.

Input:
row: int - Between 0 and 14."""
        assert type(row) in (int,float) and 0 <= row <= 15 or row == None
        if row != None:
            if self.prevmpr != None: self.basicSetMProw(self.prevmpr,0,False)
            else:
                for i in range(15):
                    self.basicSetMProw(i,0,False)
            self.prevmpr = row
            self.basicSetMProw(row,1,False)
        else:
            if self.prevmpr != None: self.basicSetMProw(self.prevmpr,0,False)
            else:
                for i in range(15):
                    self.basicSetMProw(i,0,False)
            self.prevmpr = row
    def basicSetMProw(self,row,level,fromusr=True):
        """Set a row of the MiniProgramm-LEDs.

Input:
row: int - Between 0 and 14.
level: bool - True for LEDs on; False for LEDs off."""
        assert type(row) in (int,float) and 0 <= row <= 15
        assert type(level) in (bool,int) and level in (True,False,1,0)
        assert type(fromusr) in (bool,int) and fromusr in (True,False,1,0)
        if fromusr: self.prevmpr = None
        if row != 15:
            t = ((1,1),(2,4),(5,7),(8,8),(9,11),(12,14),
                 (15,18),(19,19),(20,22),(23,25),(26,29),
                 (30,32),(33,33),(34,36),(37,40))
            for i in range(t[row][0],t[row][1]+1):
                self._set(5,i,level)
        #self.ov.update()
            
    def setZeroChk(self,level):
        "Set the LED of the Zero-Ceck to level (on 'BR')."
        self._set(2,10,level)
        #self.ov.update()
    def setTakt(self,level):
        "Set the LED of the clock-LED to level (on 'HS')."
        self._set(4,9,level)
        #self.ov.update()

    def setLoad(self,board,level):
        """Set the Load-LED to level.
Board must be an int between 0 and 5 but 4"""
        assert board in (0,1,2,3,5)
        part_nr = {0:11,1:0,2:9,3:0,5:0}[board]
        self._set(board,part_nr,level)
        #self.ov.update()
    def setEnable(self,board,level):
        """Set the Enable-LED to level.
Board must be an int between 0 and 5"""
        assert board in range(6)
        part_nr = (12,1,8,1,0,45)[board]
        self._set(board,part_nr,level)
        #self.ov.update()
        
    def _setByte(self,board,Bit0,value,half=False):
        """Intern function.
Sets a row of 8 parts on a SwitchCanvas from Bit0 to Bit0-7
if half is True: Sets a row of 4 parts on a SwitchCanvas from Bit0 to Bit0-3"""
        if half: assert 0<=value<=15
        else: assert 0<=value<=255
        b = bin(value)[2:]
        for i in range(len(b)):
            self._set(board,Bit0-i,int(b[-i-1]))
        for i in range(i+1,4 if half else 8):
            self._set(board,Bit0-i,0)

    def _set(self,board,part,level):
        """Intern function.
Set the part of the board to the level (on DetailView as well as on OverView."""
        self.ov.set(board,part,level)
        self.dv.set(board,part,level)

    def mainloop(self):
        "Stars the mainloop of view."
        self.ov.mainloop()

    def update(self):
        "Updates View."
        self.ov.update()

    def _exit(self):
        """Intern use only.
Handles the Close-Callback of the Window."""
        if self.cbE != None:
            self.cbE()
        self.destroy()
    def destroy(self):
        """Closes all windows of view."""
        self.dv.destroy()
        self.ov.destroy()

from time import sleep
if __name__ == '__main__':
    v = View(print)
    for i in (0,1,2,3,5):
        print('setByte1',i)
        v.setByte1(i,255)
    for i in (0,5):
        print('setByte2',i)
        v.setByte2(i,255)
    for i in (2,5):
        print('HalfByte',i)
        v.setHalfByte(i,15)
    for i in range(15):
        print('basicSetMProw',i)
        v.basicSetMProw(i,1)
    print('setZC')
    v.setZeroChk(1)
    print('setBT')
    v.setTakt(1)
    for i in (0,1,2,3,5):
        print('setLoad',i)
        v.setLoad(i,1)
    for i in range(6):
        print('setEnable',i)
        v.setEnable(i,1)
    v.update()
    for i in range(50):
        sleep(0.05)
        v.update()
    for i in range(15):
        print('basicSetMProw',i)
        v.basicSetMProw(i,0)
    for i in range(50):
        sleep(0.05)
        v.update()
    for i in range(15):
        v.setMProw(i)
        for i in range(20):
            sleep(0.05)
            v.update()
    for i in range(14,-1,-1):
        v.setMProw(i)
        for i in range(10):
            sleep(0.05)
            v.update()
    v.mainloop()
